## ----options, include=FALSE, cache=FALSE---------------------------------
 opts_chunk[["set"]](fig.path='fig/fig-', cache.path='cache/', fig.align='center', dev='tikz', external=FALSE, fig.width=4.5, fig.height=3, fig.show='hold', echo=FALSE, results='hide', warning=FALSE, error=FALSE, message=FALSE, tidy=FALSE,cache=TRUE,autodep=TRUE, par=TRUE, comment=NA, keep.blank.line=FALSE, tidy=FALSE)
 knit_hooks[["set"]](par=function(before, options, envir){
 if (before) { options(max.print=40) ; options(width=80) }
 if (before && options[["fig.show"]]!='none') par(mar=c(4,4,.1,.1),cex.main=1,font.main=1,cex.lab=.95,cex.axis=.9,mgp=c(2,.7,0),tcl=-.3)
 }, crop=hook_pdfcrop)
opts_chunk[["set"]](dev.args=list(pointsize=13))
options(tikzMetricsDictionary="tikz.metrics")

## ----init,cache=FALSE----------------------------------------------------
library(plyr)
#
library(tidyverse)
ggCache=FALSE
t<-theme_get()
t[["legend.title"]]<-element_blank() # <- to title for legend
theme_set(t)
#
library(lme4)
library(xtable)
options(xtable.floating=FALSE)
options(xtable.sanitize.text.function=function(x) x)
library(parallel)
library(boot)
options(boot.ncpus=detectCores())

## ----data,cache=TRUE-----------------------------------------------------
library("zTree")
files<-grep("^(Session|Pilot).*",list.files(".","*.xls",recursive=TRUE),value=TRUE)
DELEGATION<-zTreeTables(files)
subjects<-DELEGATION$subjects
globals<-DELEGATION$globals
#merging subjects and globals (to define "effective" senders used in the analysis)
subjects<-merge(subjects,globals, by=c("Date", "Period"), suffixes=c("",".y"))
#option "suffix" differentiates columnnames and assures the column names in subjects are unchanged
#recode variables
subjects<-within(subjects,{
    Date<-as.factor(Date)
    Condition<-as.factor(Condition)
    levels(Condition)<-c("no conflict","conflict")
    Sender1_Delegation<-Sender1_Delegation==1
    Sender2_Delegation<-Sender2_Delegation==1
    Sender_Delegation<-ifelse(Type==1,Sender1_Delegation,ifelse(Type==2,Sender2_Delegation,NA))
    Other_Delegation<-ifelse(Type==1,Sender2_Delegation,ifelse(Type==2,Sender1_Delegation,NA))
    Truth<-Truth==1
    Follow<-Follow==1
    sid<-as.factor(sprintf("%s-%d", Date, Subject))
    Effective_Sender<-ifelse(Type==3,NA,ifelse(Type==ceiling(SenderDraw*2),TRUE,FALSE))
    Responsible<-ifelse(Effective_Sender,!Sender_Delegation,Other_Delegation)
     })
ssOther<-subjects[,c("Date","Period","Group","Type","Truth")]
ssOther<-within(ssOther,{Type<-3-Type;OtherTruth<-Truth;Truth<-NULL})
subjects<-merge(subjects,ssOther,all.x=TRUE)

## ----Somedata for summary------------------------------------------------
subset(subjects,Type!=3) %>%
    group_by(Condition, Sender_Delegation) %>%
    summarize(`$n$`=length(Truth),
              truth=sum(Truth),
              lies=sum(!Truth),
              `mean share`=mean(Share[Profit1>0]),
              `stddev share`=sd(Share[Profit1>0])) %>%
    rename(delegate = Sender_Delegation) %>%
    mutate(delegate = ifelse(delegate,"yes","no")) -> sum.tab

## ----substat,results='asis'----------------------------------------------
print(xtable(sum.tab,digits=c(rep(0,6),2,2)),include.rownames=FALSE)

## ----TruthRatebyConditionStats-------------------------------------------
nSid<-with(subset(subjects, Type!=3),length(unique(sid)))

## ----TruthRatebyCondition,cache=ggCache----------------------------------
filter(subjects, Type!=3) %>%
    group_by(Condition,Period) %>%
    summarise(Truth=mean(Truth)) %>% 
    ggplot(aes(Period,Truth)) +
    geom_point(aes(shape=Condition)) +
    geom_line(aes(linetype=Condition)) +
    ylab("Relative frequency of truth-telling") +
    coord_cartesian(ylim=c(0,1)) + scale_shape(solid=FALSE)

## ----truthMean14---------------------------------------------------------
subset(subjects, Type!=3) %>% 
    group_by(Condition) %>%
    summarize(Truth=round(100*mean(Truth),0)) -> truth_mean14

## ----delegationFreq_plot-------------------------------------------------
#Comparing Delegation Frequencies in Condition1 and Condition2: Hypothesis hyp:SR1#
n_c<-with(subset(subjects, Condition=='conflict'&Type!=3),length(unique(sid)))

## ----delegationFreq_plot_gg,cache=ggCache--------------------------------
subjects %>% 
    group_by(Condition,Period) %>%
    summarize(delegation=mean(Sender_Delegation,na.rm=TRUE)) %>%
    ggplot(aes(Period,delegation)) +
    geom_point(aes(shape=Condition)) + 
    geom_line(aes(linetype=Condition)) + 
    ylab("Relative frequency of delegation") + 
    coord_cartesian(ylim=c(0,1)) + scale_shape(solid=FALSE)

## ----delegationFreq_test-------------------------------------------------
#assigning variables to be used by \Sexpr later in text
meanDelegate<-with(subjects, mean(Sender_Delegation, na.rm=TRUE)) 
meanDelegate_c<-with(subjects, aggregate(Sender_Delegation, list(Condition=Condition), mean, na.rm=TRUE)) 
deleNC<-round(subset(meanDelegate_c,Condition=="no conflict")[["x"]]*100)
deleC<-round(subset(meanDelegate_c,Condition=="conflict")[["x"]]*100)

## ----estDelfreq,cache=TRUE-----------------------------------------------
bsSize<-500  ### <- should be 500
set.seed(123)
estDelfreq.mer<-with(subset(subjects, Type!=3),glmer(Sender_Delegation~Condition+as.factor(Period)+(1|sid)+(1|Date),family=binomial(link=logit)))
estDelfreq.boot<-bootMer(estDelfreq.mer,fixef,nsim=bsSize,parallel="multicore")
Delfreq.ci<-boot.ci(estDelfreq.boot,index=2,type="perc")$percent[4:5]

## ----delegationFreqTrueLie_plot------------------------------------------
subset(subjects,Condition=="conflict") %>%
    group_by(Truth) %>% 
    summarize(delegation=round(100*mean(Sender_Delegation,na.rm=TRUE),0)) ->
    dtxx14

## ----DelegateByTruth,cache=ggCache---------------------------------------
subset(subjects,Condition=="conflict") %>%
    mutate(truthtelling=factor(ifelse(Truth,"truth-tellers","liars"),levels=c("truth-tellers","liars"))) %>% 
    group_by(truthtelling,Period) %>% 
    summarize(delegation=mean(Sender_Delegation,na.rm=TRUE)) %>%
    ggplot(aes(Period,delegation,shape=truthtelling,linetype=truthtelling)) +
    geom_point() +
    geom_line() + 
    ylab("Relative frequency of delegation") + 
    coord_cartesian(ylim=c(0,1)) + scale_shape(solid=FALSE)

## ----estSRDel,cache=TRUE-------------------------------------------------
## bsSize<-500
set.seed(123)
estSRDel<-with(subset(subjects, Type!=3 & Condition=='conflict'),glmer(Sender_Delegation~Truth+as.factor(Period)+(1|sid)+(1|Date),family=binomial(link=logit)))
estSRDel.boot<-bootMer(estSRDel,fixef,nsim=bsSize,parallel="multicore")
SRDel.ci<-boot.ci(estSRDel.boot,index=2,type="perc")$percent[4:5]

## ----Compensatory_stat---------------------------------------------------
richSender<-subset(subjects, Type!=3&Profit1>0)
#per cent of senders sharing positive amounts:
share_fraction<-with(richSender,round(mean(Share>0)*100,2))
# average sharing by those who shared something:
average_share<-with(richSender,round(mean(Share[Share>0]),2))
  
# average sharing as % of earnings:
average_share_percent<-with(richSender,round(mean(Share[Share>0])/80*100,1))

## ----SharingbyCondition_plot,fig.width=6,cache=ggCache-------------------
subset(subjects, Type!=3&Profit1>0) %>% 
    mutate(sEff=factor(ifelse(Effective_Sender,"effective sender","ineffective sender"),levels=c("ineffective sender","effective sender")),
            truthtelling=factor(ifelse(Truth,"truth","lies"),levels=c("truth","lies"))) %>%
    group_by(Condition,truthtelling,sEff,Period) %>% 
    summarise(Share=mean(Share),n=length(Type)) %>%
    ggplot(mapping=aes(x=Period,y=Share)) +
    geom_line(aes(linetype=truthtelling)) + 
    geom_point(aes(shape=truthtelling,size=n))+ scale_shape(solid=FALSE)+
    ylab("Average amount shared / [ECU]") +
    facet_wrap(~Condition+sEff,nrow=1) 

## ----SharingbyCondition_test,cache=FALSE---------------------------------
## bsSize<-500
set.seed(1)
mySum<-function(.) c(beta=fixef(.),sigma=sqrt(unlist(VarCorr(.))))

mer2df <- function(mer,nsim=bsSize,conf=.95) {
    b<-bootMer(mer[["mer"]],mySum,nsim=nsim,parallel="multicore")
    (b$t)
    btab0 <- t(sapply(as.list(seq(length(b$t0))),
                    function(i) boot.ci(b,index=i,conf=conf, type="perc")$percent))
    btab <- btab0[,4:5]
    rownames(btab) <- names(b$t0)
    a <- (1 - conf)/2
    a <- c(a, 1 - a)
    pct <- stats:::format.perc(a, 3)
    colnames(btab) <- pct
    betaCI<-cbind(beta=b$t0,btab)
    ## now add the fe model
    i<-grep("^XXX",names(coef(mer[["fe"]])))
    fixef<-c(coef(mer[["fe"]])[i],confint(mer[["fe"]])[i,])
    data.frame(rbind(apply(rbind(betaCI,fixef),1,function(x) sprintf("\\AA{%.2f}{%.2f}{%.2f}",x[1],x[2],x[3]))),N=nrow(mer[["mer"]]@frame))
}

## ----SharingbyCondition_mer,cache=TRUE-----------------------------------
myMer <- function(XX,data) {
    mer<-lmer(paste("Share ~ as.factor(Period) + ",XX," - 1 + (1|sid) + (1|Date)"),data=subset(data,Type!=3&Profit1>0))
    data$XXX<-data[[XX]]
    fe<-with(subset(data,Type!=3&Profit1>0),lm(Share ~ as.factor(Period) + XXX - 1 + sid))
    list(mer=mer,fe=fe)
}
##
myMer2 <- function(XX,YY,data) {
    mer<-lmer(paste("Share ~ as.factor(Period) + ",XX," + ",YY," + ",XX,"*",YY," - 1 + (1|sid) + (1|Date)"),data=subset(data,Type!=3&Profit1>0))
    data$XXX<-data[[XX]]
    data$YYY<-data[[YY]]
    fe<-with(subset(data,Type!=3&Profit1>0),lm(Share ~ as.factor(Period) + XXX + YYY + XXX*YYY - 1 + sid))
    list(mer=mer,fe=fe)
}                                                
##
mers<-list(allEq3=myMer("Truth",subjects),
           effEq3=myMer("Truth",subset(subjects,Effective_Sender)),
           allEq4=myMer("Condition",subjects),
           effEq4=myMer("Condition",subset(subjects,Effective_Sender)),
          allEq5=myMer2("Truth", "Condition", subjects),
          effEq5=myMer2("Truth", "Condition", subset(subjects, Effective_Sender))
                        )

mers.df<-rbind.fill(lapply(mers,mer2df,nsim=bsSize))
  

## ----SharingbyCondition_mer4,cache=TRUE----------------------------------
mer4<-list(
    allEq6=myMer("Sender_Delegation",subset(subjects,Condition=="conflict")),
    effEq6=myMer("Sender_Delegation",subset(subjects,Condition=="conflict"&Effective_Sender)))

mer4.df <- rbind.fill(lapply(mer4,mer2df,nsim=bsSize))

## ----SharingByConditionOut,results='asis',cache=FALSE--------------------
q<-1
mers.tab<-t(mers.df)
colnames(mers.tab)<-names(mers)
mers.tab<-mers.tab[order(rownames(mers.tab)),]
mers.tab<-mers.tab[-grep("Period",rownames(mers.tab)),]
##
bTrans<-read.csv(text="
var,tex
beta.Conditionconflict,$\\\\beta_{\\\\conflict{}}$
beta.TruthTRUE.Conditionconflict,$\\\\beta_{\\\\truth\\\\times\\\\conflict{}}$
beta.TruthTRUE,$\\\\beta_{\\\\truth}$
fixef,($\\\\beta^{\\\\text{FE}}$)
N,$N$
sigma.Date,$\\\\sigma_g$
sigma.sid,$\\\\sigma_i$")
cTrans<-read.csv(text="
var,tex
allEq3,all Eq.(\\\\ref{eq:shareT})
effEq3,effective Eq.(\\\\ref{eq:shareT})
allEq4,all Eq.(\\\\ref{eq:shareC})
effEq4,effective Eq.(\\\\ref{eq:shareC})
allEq5,all Eq.(\\\\ref{eq:shareTC})
effEq5,effective Eq.(\\\\ref{eq:shareTC})"
)
##
for(i in 1:nrow(bTrans)) rownames(mers.tab)<-sub(bTrans[i,1],bTrans[i,2],rownames(mers.tab))
for(i in 1:nrow(cTrans)) colnames(mers.tab)<-sub(cTrans[i,1],cTrans[i,2],colnames(mers.tab))
xtable(mers.tab,align=paste(c("r",rep("c",ncol(mers.tab))),collapse=""))

## ----SharingbyDelegation_plot,fig.width=6,cache=ggCache------------------
subset(subjects, Type!=3&Profit1>0) %>% 
    mutate(delegate=ifelse(Sender_Delegation,"delegate","don't delegate"),
           truthtelling=ifelse(Truth,"truth","lies")) %>%
    group_by(delegate,truthtelling,Condition,Period) %>% 
    summarise(Share=mean(Share),n=length(Type)) %>% 
    ggplot(mapping=aes(x=Period,y=Share)) +
    geom_line(mapping=aes(linetype=delegate)) + 
    geom_point(aes(shape=delegate,size=n))+ scale_shape(solid=FALSE) +
    ylab("Average amount shared / [ECU]") +
    facet_wrap(~Condition+truthtelling,nrow=1)

## ----SharingbyDelegation_test,results='asis',cache=FALSE-----------------
mer4.tab<-t(mer4.df)
colnames(mer4.tab)<-names(mer4)
mer4.tab<-mer4.tab[order(rownames(mer4.tab)),]
mer4.tab<-mer4.tab[-grep("Period",rownames(mer4.tab)),]
##
b4Trans<-read.csv(text="
var,tex
beta.Sender_DelegationTRUE,$\\\\beta_{\\\\delegation}$
fixef,($\\\\beta^{\\\\text{FE}}$)
N,$N$
sigma.Date,$\\\\sigma_g$
sigma.sid,$\\\\sigma_i$")
c4Trans<-read.csv(text="
var,tex
allEq6,all conflict Eq.(\\\\ref{eq:shareDel})
effEq6,effective conflict Eq.(\\\\ref{eq:shareDel})
noconfEq4,conf.del./no conf.
lieEq4,lie delegation")
##
for(i in 1:nrow(b4Trans)) rownames(mer4.tab)<-sub(b4Trans[i,1],b4Trans[i,2],rownames(mer4.tab))
for(i in 1:nrow(c4Trans)) colnames(mer4.tab)<-sub(c4Trans[i,1],c4Trans[i,2],colnames(mer4.tab))

xtable(mer4.tab,align=paste(c("r",rep("c",ncol(mer4.tab))),collapse=""))

## ----eval=TRUE-----------------------------------------------------------
save.image("image.Rdata")

## ----mer2la--------------------------------------------------------------
mer2la <- function(mer,replace=NULL,drop=NULL,sep=NULL) {
    nf<-function(x) x
    mer.s <- summary(mer)
    fixef<-mer.s[["coefficients"]]
    for (i in drop) rownames(fixef)<-gsub(i,"",rownames(fixef))
    rownames(fixef)<-revalue(rownames(fixef,replace))
    #
    varcor<-data.frame()
    ranef<-rbind.fill(lapply(names(mer.s[["varcor"]]),function(n) {
        s<-attr(mer.s[["varcor"]][[n]],"stddev")
        data.frame(list(Groups=n,Name=names(s)[1],Stddev=s,n=mer.s[["ngrps"]][n]))
    }))
    ranef<-rbind.fill(ranef,data.frame(list(Groups="Residuals",Name="(Intercept)",Stddev=sd(mer.s[["residuals"]]),n=length(mer.s[["residuals"]]))))
    ranef<-within(ranef,{
        Groups<-revalue(Groups,replace)
        Name<-revalue(Name,replace)
    })
    di<-c(rep(min(4-floor(log10(abs(fixef[,"Estimate"])))),3),4,5)
    fixef.xt<-xtable(fixef,digits=di)
    colnames(fixef.xt)[1]<-"\\(\\beta\\)"
    colnames(fixef.xt)[2]<-"\\(\\sigma\\)"
    colnames(fixef.xt)[3]<-"\\(z\\)"
    colnames(fixef.xt)[4]<-"\\(\\text{Pr}(>|z|)\\)"
    print(fixef.xt,sanitize.rownames.function=nf,include.rownames=TRUE)
    di<-c(rep(min(4-floor(log10(ranef[,"Stddev"]))),4),0)
    ranef.xt<-xtable(ranef,digits=di)
    colnames(ranef.xt)[3:4]<-c("\\(\\sigma\\)","\\(n\\)")
    cat(sep)
    print(ranef.xt,include.rownames=FALSE)
    list(fixef=fixef,ranef=ranef)
}

## ----estDelfreq.mer,results='asis'---------------------------------------
q<-mer2la(estDelfreq.mer,replace=c(sid="$i$",Date="$g$"),drop=c("as.factor","Condition"),sep="\\end{center}\\endgraf Random effects:\\endgraf\\begin{center}")

## ----estSRDel.tab,results='asis'-----------------------------------------
q<-mer2la(estSRDel,replace=c(sid="$i$",Date="$g$"),drop=c("as.factor","Condition","TRUE"),sep="\\end{center}\\endgraf Random effects:\\endgraf\\begin{center}")

